<?php
/**
 * Theme customizer
 *
 * @package Ronmi
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Ronmi_Customize {
	/**
	 * Customize settings
	 *
	 * @var array
	 */
	protected $config = array();

	/**
	 * The class constructor
	 *
	 * @param array $config
	 */
	public function __construct( $config ) {
		$this->config = $config;

		if ( ! class_exists( 'Kirki' ) ) {
			return;
		}

		$this->register();
	}

	/**
	 * Register settings
	 */
	public function register() {

		/**
		 * Add the theme configuration
		 */
		if ( ! empty( $this->config['theme'] ) ) {
			Kirki::add_config(
				$this->config['theme'], array(
					'capability'  => 'edit_theme_options',
					'option_type' => 'theme_mod',
				)
			);
		}

		/**
		 * Add panels
		 */
		if ( ! empty( $this->config['panels'] ) ) {
			foreach ( $this->config['panels'] as $panel => $settings ) {
				Kirki::add_panel( $panel, $settings );
			}
		}

		/**
		 * Add sections
		 */
		if ( ! empty( $this->config['sections'] ) ) {
			foreach ( $this->config['sections'] as $section => $settings ) {
				Kirki::add_section( $section, $settings );
			}
		}

		/**
		 * Add fields
		 */
		if ( ! empty( $this->config['theme'] ) && ! empty( $this->config['fields'] ) ) {
			foreach ( $this->config['fields'] as $name => $settings ) {
				if ( ! isset( $settings['settings'] ) ) {
					$settings['settings'] = $name;
				}

				Kirki::add_field( $this->config['theme'], $settings );
			}
		}
	}

	/**
	 * Get config ID
	 *
	 * @return string
	 */
	public function get_theme() {
		return $this->config['theme'];
	}

	/**
	 * Get customize setting value
	 *
	 * @param string $name
	 *
	 * @return bool|string
	 */
	public function get_option( $name ) {

		$default = $this->get_option_default( $name );

		return get_theme_mod( $name, $default );
	}

	/**
	 * Get default option values
	 *
	 * @param $name
	 *
	 * @return mixed
	 */
	public function get_option_default( $name ) {
		if ( ! isset( $this->config['fields'][ $name ] ) ) {
			return false;
		}

		return isset( $this->config['fields'][ $name ]['default'] ) ? $this->config['fields'][ $name ]['default'] : false;
	}
}

/**
 * This is a short hand function for getting setting value from customizer
 *
 * @param string $name
 *
 * @return bool|string
 */
function ronmi_get_option( $name ) {
	global $ronmi_customize;

	$value = false;

	if ( class_exists( 'Kirki' ) ) {
		$value = Kirki::get_option( 'ronmi', $name );
	} elseif ( ! empty( $ronmi_customize ) ) {
		$value = $ronmi_customize->get_option( $name );
	}

	return apply_filters( 'ronmi_get_option', $value, $name );
}

/**
 * Get default option values
 *
 * @param $name
 *
 * @return mixed
 */
function ronmi_get_option_default( $name ) {
	global $ronmi_customize;

	if ( empty( $ronmi_customize ) ) {
		return false;
	}

	return $ronmi_customize->get_option_default( $name );
}

/**
 * Move some default sections to `general` panel that registered by theme
 *
 * @param object $wp_customize
 */
function ronmi_customize_modify( $wp_customize ) {
	$wp_customize->get_section( 'title_tagline' )->panel     = 'general';
	$wp_customize->get_section( 'static_front_page' )->panel = 'general';
}

add_action( 'customize_register', 'ronmi_customize_modify' );


/**
 * Get customize settings
 *
 * Priority (Order) WordPress Live Customizer default: 
 * @link https://developer.wordpress.org/themes/customize-api/customizer-objects/
 *
 * @return array
 */
function ronmi_customize_settings() {
	/**
	 * Customizer configuration
	 */

	$settings = array(
		'theme' => 'ronmi',
	);

	$panels = array(
		'general'     => array(
			'priority' => 5,
			'title'    => esc_html__( 'General', 'ronmi' ),
		),
        'blog'        => array(
            'title'      => esc_html__( 'Blog', 'ronmi' ),
            'priority'   => 10,
            'capability' => 'edit_theme_options',
        ),
        'portfolio'       => array(
            'title'       => esc_html__( 'Portfolio', 'ronmi' ),
            'priority'    => 10,
            'capability'  => 'edit_theme_options',          
        ),
	);

	$sections = array(
        /*header*/
        'main_header'     => array(
            'title'       => esc_html__( 'Header', 'ronmi' ),
            'description' => '',
            'priority'    => 8,
            'capability'  => 'edit_theme_options',
        ),
        /*page header*/
        'page_header'     => array(
            'title'       => esc_html__( 'Page Header', 'ronmi' ),
            'description' => '',
            'priority'    => 9,
            'capability'  => 'edit_theme_options',
        ),
        /* blog */
        'blog_page'           => array(
            'title'       => esc_html__( 'Blog Page', 'ronmi' ),
            'description' => '',
            'priority'    => 10,
            'capability'  => 'edit_theme_options',
            'panel'       => 'blog',
        ),
        'single_post'           => array(
            'title'       => esc_html__( 'Single Post', 'ronmi' ),
            'description' => '',
            'priority'    => 10,
            'capability'  => 'edit_theme_options',
            'panel'       => 'blog',
        ),
        // footer
        'footer'         => array(
            'title'      => esc_html__( 'Footer', 'ronmi' ),
            'priority'   => 10,
            'capability' => 'edit_theme_options',
        ),
        /* portfolio */
        'portfolio_page'  => array(
            'title'       => esc_html__( 'Archive Page', 'ronmi' ),
            'priority'    => 10,
            'capability'  => 'edit_theme_options',
            'panel'       => 'portfolio',           
        ),
        'portfolio_post'  => array(
            'title'       => esc_html__( 'Single Page', 'ronmi' ),
            'priority'    => 10,
            'capability'  => 'edit_theme_options',
            'panel'       => 'portfolio',           
        ),
		/* typography */
		'typography'           => array(
            'title'       => esc_html__( 'Typography', 'ronmi' ),
            'description' => '',
            'priority'    => 15,
            'capability'  => 'edit_theme_options',
        ),
		/* 404 */
		'error_404'       => array(
            'title'       => esc_html__( '404', 'ronmi' ),
            'description' => '',
            'priority'    => 11,
            'capability'  => 'edit_theme_options',
        ),
        /* color scheme */
        'color_scheme'   => array(
			'title'      => esc_html__( 'Color Scheme', 'ronmi' ),
			'priority'   => 200,
			'capability' => 'edit_theme_options',
		),
		/* js code */
		'script_code'   => array(
			'title'      => esc_html__( 'Google Analytics(Script Code)', 'ronmi' ),
			'priority'   => 210,
			'capability' => 'edit_theme_options',
		),
	);

	$fields = array(
        /* header settings */
        'header_position'   => array(
            'type'        => 'select',  
            'label'       => esc_attr__( 'Select Header Position', 'ronmi' ), 
            'description' => esc_attr__( 'Choose the header position on desktop.', 'ronmi' ),
            'placeholder' => esc_html__( 'Choose an option', 'ronmi' ),
            'section'     => 'main_header', 
            'default'     => 'header_top', 
            'priority'    => 1,
            'choices'     => array(
                'header_top' => esc_attr__( 'Header Top', 'ronmi' ),
                'header_left' => esc_attr__( 'Header Left', 'ronmi' ),
            ),
        ),
        'width_lheader'   => array(
            'type'     => 'number',
            'label'    => esc_html__( 'Header Width', 'ronmi' ), 
            'section'  => 'main_header',
            'priority' => 2,
            'default'  => 340,
            'output'    => array(
                array(
                    'element'  => '.site-header-vertical',
                    'property' => 'width',
                    'units'    => 'px',
                    'media_query' => '@media (min-width: 1025px)',
                ),
                array(
                    'element'  => '.header-vertical .site-content, .header-vertical .site-footer',
                    'property' => 'margin-left',
                    'units'    => 'px',
                    'media_query' => '@media (min-width: 1025px)',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'header_position',
                    'operator' => '==',
                    'value'    => 'header_left',
                ),
            ),
        ),
        'header_layout'   => array(
            'type'        => 'select',  
            'label'       => esc_attr__( 'Select Header Desktop', 'ronmi' ), 
            'description' => esc_attr__( 'Choose the header on desktop.', 'ronmi' ), 
            'section'     => 'main_header', 
            'default'     => '', 
            'priority'    => 3,
            'placeholder' => esc_attr__( 'Select a header', 'ronmi' ), 
            'choices'     => ( class_exists( 'Kirki_Helper' ) ) ? Kirki_Helper::get_posts( array( 'post_type' => 'ot_header_builders', 'posts_per_page' => -1 ) ) : array(),
        ),
        'header_fixed'    => array(
            'type'        => 'toggle',
            'label'       => esc_html__( 'Header Transparent?', 'ronmi' ),
            'description' => esc_attr__( 'Enable when your header is transparent.', 'ronmi' ), 
            'section'     => 'main_header',
            'default'     => '',
            'priority'    => 4,
            'active_callback' => array(
                array(
                    'setting'  => 'header_position',
                    'operator' => '==',
                    'value'    => 'header_top',
                ),
            ),
        ),
        'header_mobile'   => array(
            'type'        => 'select',  
            'label'       => esc_attr__( 'Select Header Mobile', 'ronmi' ), 
            'description' => esc_attr__( 'Choose the header on mobile.', 'ronmi' ), 
            'section'     => 'main_header', 
            'default'     => '', 
            'priority'    => 5,
            'placeholder' => esc_attr__( 'Select a header', 'ronmi' ), 
            'choices'     => ( class_exists( 'Kirki_Helper' ) ) ? Kirki_Helper::get_posts( array( 'post_type' => 'ot_header_builders', 'posts_per_page' => -1 ) ) : array(),
        ),
        'sidepanel_layout'     => array(
            'type'        => 'select',  
            'label'       => esc_attr__( 'Select Side Panel', 'ronmi' ), 
            'description' => esc_attr__( 'Choose the side panel on header.', 'ronmi' ), 
            'section'     => 'main_header', 
            'default'     => '', 
            'priority'    => 6,
            'placeholder' => esc_attr__( 'Select a panel', 'ronmi' ), 
            'choices'     => ( class_exists( 'Kirki_Helper' ) ) ? Kirki_Helper::get_posts( array( 'post_type' => 'ot_header_builders', 'posts_per_page' => -1 ) ) : array(),
            'active_callback' => array(
                array(
                    'setting'  => 'header_position',
                    'operator' => '==',
                    'value'    => 'header_top',
                ),
            ),
        ),
        'panel_left'     => array(
            'type'        => 'toggle',
            'label'       => esc_html__( 'Side Panel On Left', 'ronmi' ),
            'section'     => 'main_header',
            'default'     => '0',
            'priority'    => 7,
            'active_callback' => array(
                array(
                    'setting'  => 'header_position',
                    'operator' => '==',
                    'value'    => 'header_top',
                ),
            ),
        ), 
        /*page header */
        'pheader_switch'  => array(
            'type'        => 'toggle',
            'label'       => esc_html__( 'Page Header On/Off', 'ronmi' ),
            'section'     => 'page_header',
            'default'     => 1,
            'priority'    => 10,
        ),
        'breadcrumbs'     => array(
            'type'        => 'toggle',
            'label'       => esc_html__( 'Breadcrumbs On/Off', 'ronmi' ),
            'section'     => 'page_header',
            'default'     => 1,
            'priority'    => 10,
            'active_callback' => array(
                array(
                    'setting'  => 'pheader_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'pheader_align'    => array(
            'type'     => 'radio',
            'label'    => esc_html__( 'Text Align', 'ronmi' ),
            'section'  => 'page_header',
            'default'  => 'text-center',
            'priority' => 10,
            'choices'     => array(
                'text-center'   => esc_html__( 'Center', 'ronmi' ),
                'text-left'     => esc_html__( 'Left', 'ronmi' ),
                'text-right'    => esc_html__( 'Right', 'ronmi' ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'pheader_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'pheader_img'  => array(
            'type'     => 'image',
            'label'    => esc_html__( 'Background Image', 'ronmi' ),
            'section'  => 'page_header',
            'default'  => '',
            'priority' => 10,
            'output'    => array(
                array(
                    'element'  => '.page-header',
                    'property' => 'background-image'
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'pheader_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'pheader_color'    => array(
            'type'     => 'color',
            'label'    => esc_html__( 'Background Color', 'ronmi' ),
            'section'  => 'page_header',
            'priority' => 10,
            'output'    => array(
                array(
                    'element'  => '.page-header',
                    'property' => 'background-color'
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'pheader_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'ptitle_color'    => array(
            'type'     => 'color',
            'label'    => esc_html__( 'Title Color', 'ronmi' ),
            'section'  => 'page_header',
            'priority' => 10,
            'output'    => array(
                array(
                    'element'  => '.page-header .page-title',
                    'property' => 'color'
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'pheader_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'bread_color'    => array(
            'type'     => 'color',
            'label'    => esc_html__( 'Breadcrumbs Color', 'ronmi' ),
            'section'  => 'page_header',
            'priority' => 10,
            'output'    => array(
                array(
                    'element'  => '.page-header .breadcrumbs li, .page-header .breadcrumbs li a, .page-header .breadcrumbs li a:hover, .page-header .breadcrumbs li:before',
                    'property' => 'color'
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'pheader_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
                array(
                    'setting'  => 'breadcrumbs',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'pheader_height'  => array(
            'type'     => 'dimensions',
            'label'    => esc_html__( 'Page Header Height (Ex: 300px)', 'ronmi' ),
            'section'  => 'page_header',
            'transport' => 'auto',
            'priority' => 10,
            'choices'   => array(
                'desktop' => esc_attr__( 'Desktop', 'ronmi' ),
                'tablet'  => esc_attr__( 'Tablet', 'ronmi' ),
                'mobile'  => esc_attr__( 'Mobile', 'ronmi' ),
            ),
            'output'   => array(
                array(
                    'choice'      => 'mobile',
                    'element'     => '.page-header, .single-post .page-header',
                    'property'    => 'height',
                    'media_query' => '@media (max-width: 767px)',
                ),
                array(
                    'choice'      => 'tablet',
                    'element'     => '.page-header, .single-post .page-header',
                    'property'    => 'height',
                    'media_query' => '@media (min-width: 768px) and (max-width: 1024px)',
                ),
                array(
                    'choice'      => 'desktop',
                    'element'     => '.page-header, .single-post .page-header',
                    'property'    => 'height',
                    'media_query' => '@media (min-width: 1025px)',
                ),
            ),
            'default' => array(
                'desktop' => '',
                'tablet'  => '',
                'mobile'  => '',
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'pheader_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'padd_left_right' => array(
            'type'     => 'dimensions',
            'label'    => esc_html__( 'Padding Left/Right (Ex: 100px)', 'ronmi' ),
            'section'  => 'page_header',
            'priority' => 10,
            'default'     => array(
                'left'  => '',
                'right' => '',
            ),
            'choices'     => array(
                'left'  => esc_html__( 'Left', 'ronmi' ),
                'right'   => esc_html__( 'Right', 'ronmi' ),
            ),
            'output'    => array(
                array(
                    'choice'      => 'left',
                    'element'     => '.page-header .page-header-content',
                    'property'    => 'padding-left',
                ),
                array(
                    'choice'      => 'right',
                    'element'     => '.page-header .page-header-content',
                    'property'    => 'padding-right',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'pheader_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'head_size'  => array(
            'type'     => 'dimensions',
            'label'    => esc_html__( 'Page Title Size (Ex: 30px)', 'ronmi' ),
            'section'  => 'page_header',
            'transport' => 'auto',
            'priority' => 10,
            'choices'   => array(
                'desktop' => esc_attr__( 'Desktop', 'ronmi' ),
                'tablet'  => esc_attr__( 'Tablet', 'ronmi' ),
                'mobile'  => esc_attr__( 'Mobile', 'ronmi' ),
            ),
            'output'   => array(
                array(
                    'choice'      => 'mobile',
                    'element'     => '.page-header .page-title',
                    'property'    => 'font-size',
                    'media_query' => '@media (max-width: 767px)',
                ),
                array(
                    'choice'      => 'tablet',
                    'element'     => '.page-header .page-title',
                    'property'    => 'font-size',
                    'media_query' => '@media (min-width: 768px) and (max-width: 1024px)',
                ),
                array(
                    'choice'      => 'desktop',
                    'element'     => '.page-header .page-title',
                    'property'    => 'font-size',
                    'media_query' => '@media (min-width: 1024px)',
                ),
            ),
            'default' => array(
                'desktop' => '',
                'tablet'  => '',
                'mobile'  => '',
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'pheader_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        /* blog settings */
        'blog_layout'           => array(
            'type'        => 'radio-image',
            'label'       => esc_html__( 'Blog Layout', 'ronmi' ),
            'section'     => 'blog_page',
            'default'     => 'content-sidebar',
            'priority'    => 7,
            'description' => esc_html__( 'Select default sidebar for the blog page.', 'ronmi' ),
            'choices'     => array(
                'content-sidebar'   => get_template_directory_uri() . '/inc/backend/images/right.png',
                'sidebar-content'   => get_template_directory_uri() . '/inc/backend/images/left.png',
                'full-content'      => get_template_directory_uri() . '/inc/backend/images/full.png',
            )
        ),
        'blog_style'           => array(
            'type'        => 'select',
            'label'       => esc_html__( 'Blog Style', 'ronmi' ),
            'section'     => 'blog_page',
            'default'     => 'list',
            'priority'    => 8,
            'description' => esc_html__( 'Select style default for the blog page.', 'ronmi' ),
            'choices'     => array(
                'list' => esc_attr__( 'Blog List', 'ronmi' ),
                'grid' => esc_attr__( 'Blog Grid', 'ronmi' ),
            ),
        ),
        'blog_columns'           => array(
            'type'        => 'select',
            'label'       => esc_html__( 'Blog Columns', 'ronmi' ),
            'section'     => 'blog_page',
            'default'     => 'pf_2_cols',
            'priority'    => 8,
            'description' => esc_html__( 'Select columns default for the blog page.', 'ronmi' ),
            'choices'     => array(
                'pf_2_cols' => esc_attr__( '2 Columns', 'ronmi' ),
                'pf_3_cols' => esc_attr__( '3 Columns', 'ronmi' ),
                'pf_4_cols' => esc_attr__( '4 Columns', 'ronmi' ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'blog_style',
                    'operator' => '==',
                    'value'    => 'grid',
                ),
            ),
        ),
        'post_entry_meta'              => array(
            'type'     => 'multicheck',
            'label'    => esc_html__( 'Entry Meta', 'ronmi' ),
            'section'  => 'blog_page',
            'default'  => array( 'author', 'date', 'comm' ),
            'choices'  => array(
                'date'    => esc_html__( 'Date', 'ronmi' ),
                'author'  => esc_html__( 'Author', 'ronmi' ),
                'comm'     => esc_html__( 'Comments', 'ronmi' ),
            ),
            'priority' => 10,
        ),
        'blog_read_more'      => array(
            'type'            => 'text',
            'label'           => esc_html__( 'Details Button', 'ronmi' ),
            'section'         => 'blog_page',
            'default'         => esc_html__( 'Read More', 'ronmi' ),
            'priority'        => 11,
        ),
        'blog_prev_pagination'      => array(
            'type'            => 'text',
            'label'           => esc_html__( 'Previous pagination label', 'ronmi' ),
            'section'         => 'blog_page',
            'default'         => esc_html__( 'Prev', 'ronmi' ),
            'priority'        => 11,
        ),
        'blog_next_pagination'      => array(
            'type'            => 'text',
            'label'           => esc_html__( 'Next pagination label', 'ronmi' ),
            'section'         => 'blog_page',
            'default'         => esc_html__( 'Next', 'ronmi' ),
            'priority'        => 11,
        ),
        /* single blog */
        'single_post_layout'           => array(
            'type'        => 'radio-image',
            'label'       => esc_html__( 'Layout', 'ronmi' ),
            'section'     => 'single_post',
            'default'     => 'content-sidebar',
            'priority'    => 10,
            'choices'     => array(
                'content-sidebar'   => get_template_directory_uri() . '/inc/backend/images/right.png',
                'sidebar-content'   => get_template_directory_uri() . '/inc/backend/images/left.png',
                'full-content'      => get_template_directory_uri() . '/inc/backend/images/full.png',
            )
        ),
        'ptitle_post'               => array(
            'type'            => 'text',
            'label'           => esc_html__( 'Page Title', 'ronmi' ),
            'section'         => 'single_post',
            'default'         => '',
            'priority'        => 10,
        ),
        'related_post_label'        => array(
            'type'            => 'text',
            'label'           => esc_html__( 'Related Posts', 'ronmi' ),
            'section'         => 'single_post',
            'default'         => esc_html__( 'Related Posts', 'ronmi' ),
            'priority'        => 10,
        ),
        'single_separator1'     => array(
            'type'        => 'custom',
            'label'       => esc_html__( 'Social Share', 'ronmi' ),
            'section'     => 'single_post',
            'default'     => '<hr>',
            'priority'    => 10,
        ),
        'post_socials'              => array(
            'type'     => 'multicheck',
            'section'  => 'single_post',
            'default'  => array( 'twitter', 'facebook', 'pinterest', 'linkedin' ),
            'choices'  => array(
                'twit'      => esc_html__( 'Twitter', 'ronmi' ),
                'face'      => esc_html__( 'Facebook', 'ronmi' ),
                'pint'      => esc_html__( 'Pinterest', 'ronmi' ),
                'link'      => esc_html__( 'Linkedin', 'ronmi' ),
                'google'    => esc_html__( 'Google Plus', 'ronmi' ),
                'tumblr'    => esc_html__( 'Tumblr', 'ronmi' ),
                'reddit'    => esc_html__( 'Reddit', 'ronmi' ),
                'vk'        => esc_html__( 'VK', 'ronmi' ),
            ),
            'priority' => 10,
        ),
        'single_separator2'     => array(
            'type'        => 'custom',
            'label'       => esc_html__( 'Entry Footer', 'ronmi' ),
            'section'     => 'single_post',
            'default'     => '<hr>',
            'priority'    => 10,
        ),
        'author_box'      => array(
            'type'        => 'checkbox',
            'label'       => esc_attr__( 'Author Info Box', 'ronmi' ),
            'section'     => 'single_post',
            'default'     => true,
            'priority'    => 10,
        ),
        'post_nav'        => array(
            'type'        => 'checkbox',
            'label'       => esc_attr__( 'Post Navigation', 'ronmi' ),
            'section'     => 'single_post',
            'default'     => true,
            'priority'    => 10,
        ),
        'related_post'    => array(
            'type'        => 'checkbox',
            'label'       => esc_attr__( 'Related Posts', 'ronmi' ),
            'section'     => 'single_post',
            'default'     => true,
            'priority'    => 10,
        ),
        /* Portfolio settings */
        'portfolio_archive'           => array(
            'type'        => 'select',
            'label'       => esc_html__( 'Portfolio Archive', 'ronmi' ),
            'section'     => 'portfolio_page',
            'default'     => 'archive_default',
            'priority'    => 1,
            'description' => esc_html__( 'Select page default for the portfolio archive page.', 'ronmi' ),
            'choices'     => array(
                'archive_default' => esc_attr__( 'Archive page default', 'ronmi' ),
                'archive_custom' => esc_attr__( 'Archive page custom', 'ronmi' ),
            ),
        ),
        'archive_page_custom'     => array(
            'type'        => 'dropdown-pages',  
            'label'       => esc_attr__( 'Select Page', 'ronmi' ), 
            'description' => esc_attr__( 'Choose a custom page for archive portfolio page.', 'ronmi' ), 
            'section'     => 'portfolio_page', 
            'default'     => '', 
            'priority'    => 2,         
            'active_callback' => array(
                array(
                    'setting'  => 'portfolio_archive',
                    'operator' => '==',
                    'value'    => 'archive_custom',
                ),
            ),
        ),
        'portfolio_column'           => array(
            'type'        => 'select',
            'label'       => esc_html__( 'Portfolio Columns', 'ronmi' ),
            'section'     => 'portfolio_page',
            'default'     => '3cl',
            'priority'    => 3,
            'description' => esc_html__( 'Select default column for the portfolio page.', 'ronmi' ),
            'choices'     => array(
                '2cl' => esc_attr__( '2 Column', 'ronmi' ),
                '3cl' => esc_attr__( '3 Column', 'ronmi' ),
                '4cl' => esc_attr__( '4 Column', 'ronmi' ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'portfolio_archive',
                    'operator' => '==',
                    'value'    => 'archive_default',
                ),
            ),
        ),
        'portfolio_style'           => array(
            'type'        => 'select',
            'label'       => esc_html__( 'Hover Style', 'ronmi' ),
            'section'     => 'portfolio_page',
            'default'     => 'style-1',
            'priority'    => 4,
            'description' => esc_html__( 'Select default style for the portfolio page.', 'ronmi' ),
            'choices'     => array(
                'style-1' => esc_attr__( 'Background Solid', 'ronmi' ),
                'style-2' => esc_attr__( 'Under Image', 'ronmi' ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'portfolio_archive',
                    'operator' => '==',
                    'value'    => 'archive_default',
                ),
            ),
        ),
        'portfolio_posts_per_page' => array(
            'type'        => 'number',
            'section'     => 'portfolio_page',
            'priority'    => 5,
            'label'       => esc_html__( 'Posts per page', 'ronmi' ),          
            'description' => esc_html__( 'Change Posts Per Page for Portfolio Archive, Taxonomy.', 'ronmi' ),
            'default'     => '6',
            'active_callback' => array(
                array(
                    'setting'  => 'portfolio_archive',
                    'operator' => '==',
                    'value'    => 'archive_default',
                ),
            ),
        ),
        'pf_nav'          => array(
            'type'        => 'toggle',
            'label'       => esc_attr__( 'Projects Navigation On/Off', 'ronmi' ),
            'section'     => 'portfolio_post',
            'default'     => 1,
            'priority'    => 7,
        ),
        'pf_related_switch'     => array(
            'type'        => 'toggle',
            'label'       => esc_attr__( 'Related Projects On/Off', 'ronmi' ),
            'section'     => 'portfolio_post',
            'default'     => 1,
            'priority'    => 7,
        ),
        'pf_related_text'      => array(
            'type'            => 'text',
            'label'           => esc_html__( 'Related Projects Heading', 'ronmi' ),
            'section'         => 'portfolio_post',
            'default'         => esc_html__( 'Related Projects', 'ronmi' ),
            'priority'        => 7,
            'active_callback' => array(
                array(
                    'setting'  => 'pf_related_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        /* footer settings */
        'footer_layout'     => array(
            'type'        => 'select',  
            'label'       => esc_attr__( 'Select Footer', 'ronmi' ), 
            'description' => esc_attr__( 'Choose a footer for all site here.', 'ronmi' ), 
            'section'     => 'footer', 
            'default'     => '', 
            'priority'    => 1,
            'placeholder' => esc_attr__( 'Select a footer', 'ronmi' ), 
            'choices'     => ( class_exists( 'Kirki_Helper' ) ) ? Kirki_Helper::get_posts( array( 'post_type' => 'ot_footer_builders', 'posts_per_page' => -1 ) ) : array(),
        ),
        'backtotop_separator'     => array(
            'type'        => 'custom',
            'label'       => '',
            'section'     => 'footer',
            'default'     => '<hr>',
            'priority'    => 2,
        ),
        'backtotop'  => array(
            'type'        => 'toggle',
            'label'       => esc_html__( 'Back To Top On/Off?', 'ronmi' ),
            'section'     => 'footer',
            'default'     => 1,
            'priority'    => 3,
        ),
        'bg_backtotop'    => array(
            'type'     => 'color',
            'label'    => esc_html__( 'Background Color', 'ronmi' ),
            'section'  => 'footer',
            'priority' => 4,
            'default'     => '',
            'output'    => array(
                array(
                    'element'  => '#back-to-top',
                    'property' => 'background',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'backtotop',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'color_backtotop' => array(
            'type'     => 'color',
            'label'    => esc_html__( 'Color', 'ronmi' ),
            'section'  => 'footer',
            'priority' => 5,
            'default'     => '',
            'output'    => array(
                array(
                    'element'  => '#back-to-top > i:before',
                    'property' => 'color',
                )
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'backtotop',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'spacing_backtotop' => array(
            'type'     => 'dimensions',
            'label'    => esc_html__( 'Spacing', 'ronmi' ),
            'section'  => 'footer',
            'priority' => 6,
            'default'     => array(
                'bottom'  => '',
                'right' => '',
            ),
            'choices'     => array(
                'labels' => array(
                    'bottom'  => esc_html__( 'Bottom (Ex: 20px)', 'ronmi' ),
                    'right'   => esc_html__( 'Right (Ex: 20px)', 'ronmi' ),
                ),
            ),
            'output'    => array(
                array(
                    'choice'      => 'bottom',
                    'element'     => '#back-to-top.show',
                    'property'    => 'bottom',
                ),
                array(
                    'choice'      => 'right',
                    'element'     => '#back-to-top.show',
                    'property'    => 'right',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'backtotop',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
		/* typography */
        'typography_switch'     => array(
            'type'        => 'toggle',
            'label'       => esc_attr__( 'Typography Customize?', 'ronmi' ),
            'section'     => 'typography',
            'default'     => 0,
            'priority'    => 1,
        ),
        'body_typo'    => array(
            'type'     => 'typography',
            'label'    => esc_html__( 'Body Font', 'ronmi' ),
            'section'  => 'typography',
            'priority' => 2,
            'default'  => array(
                'font-family'    => '',
                'variant'        => 'regular',
                'font-size'      => '16px',
                'line-height'    => '1.875',
                'letter-spacing' => '0',
                'subsets'        => array( 'latin', 'latin-ext' ),
                'color'          => '#5f6160',
                'text-transform' => 'none',
            ),
            'output'      => array(
                array(
                    'element' => 'body, p, .elementor-element .elementor-widget-text-editor, .elementor-element .elementor-widget-icon-list .elementor-icon-list-item',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'typography_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'heading1_typo'=> array(
            'type'     => 'typography',
            'label'    => esc_html__( 'Heading 1', 'ronmi' ),
            'section'  => 'typography',
            'priority' => 3,
            'default'  => array(
                'font-family'    => '',
                'variant'        => '400',
                'font-size'      => '48px',
                'line-height'    => '1.4',
                'letter-spacing' => '0',
                'subsets'        => array( 'latin', 'latin-ext' ),
                'color'          => '#191d1b',
                'text-transform' => 'none',
            ),
            'output'      => array(
                array(
                    'element' => 'h1',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'typography_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'heading2_typo'=> array(
            'type'     => 'typography',
            'label'    => esc_html__( 'Heading 2', 'ronmi' ),
            'section'  => 'typography',
            'priority' => 4,
            'default'  => array(
                'font-family'    => '',
                'variant'        => '400',
                'font-size'      => '42px',
                'line-height'    => '1.4',
                'letter-spacing' => '0',
                'subsets'        => array( 'latin', 'latin-ext' ),
                'color'          => '#191d1b',
                'text-transform' => 'none',
            ),
            'output'      => array(
                array(
                    'element' => 'h2',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'typography_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'heading3_typo'=> array(
            'type'     => 'typography',
            'label'    => esc_html__( 'Heading 3', 'ronmi' ),
            'section'  => 'typography',
            'priority' => 5,
            'default'  => array(
                'font-family'    => '',
                'variant'        => '400',
                'font-size'      => '36px',
                'line-height'    => '1.4',
                'letter-spacing' => '0',
                'subsets'        => array( 'latin', 'latin-ext' ),
                'color'          => '#191d1b',
                'text-transform' => 'none',
            ),
            'output'      => array(
                array(
                    'element' => 'h3',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'typography_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'heading4_typo'=> array(
            'type'     => 'typography',
            'label'    => esc_html__( 'Heading 4', 'ronmi' ),
            'section'  => 'typography',
            'priority' => 6,
            'default'  => array(
                'font-family'    => '',
                'variant'        => '400',
                'font-size'      => '30px',
                'line-height'    => '1.4',
                'letter-spacing' => '0',
                'subsets'        => array( 'latin', 'latin-ext' ),
                'color'          => '#191d1b',
                'text-transform' => 'none',
            ),
            'output'      => array(
                array(
                    'element' => 'h4',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'typography_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'heading5_typo'=> array(
            'type'     => 'typography',
            'label'    => esc_html__( 'Heading 5', 'ronmi' ),
            'section'  => 'typography',
            'priority' => 7,
            'default'  => array(
                'font-family'    => '',
                'variant'        => '400',
                'font-size'      => '24px',
                'line-height'    => '1.4',
                'letter-spacing' => '0',
                'subsets'        => array( 'latin', 'latin-ext' ),
                'color'          => '#191d1b',
                'text-transform' => 'none',
            ),
            'output'      => array(
                array(
                    'element' => 'h5',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'typography_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),
        'heading6_typo'=> array(
            'type'     => 'typography',
            'label'    => esc_html__( 'Heading 6', 'ronmi' ),
            'section'  => 'typography',
            'priority' => 8,
            'default'  => array(
                'font-family'    => '',
                'variant'        => '400',
                'font-size'      => '20px',
                'line-height'    => '1.4',
                'letter-spacing' => '0',
                'subsets'        => array( 'latin', 'latin-ext' ),
                'color'          => '#191d1b',
                'text-transform' => 'h6',
            ),
            'output'      => array(
                array(
                    'element' => 'h6',
                ),
            ),
            'active_callback' => array(
                array(
                    'setting'  => 'typography_switch',
                    'operator' => '==',
                    'value'    => 1,
                ),
            ),
        ),

		/* 404 */
		'custom_404'      => array(
            'type'        => 'toggle',
			'label'       => esc_html__( 'Customize?', 'ronmi' ),
            'section'     => 'error_404',
			'default'     => 0,
			'priority'    => 3,
		),
		'page_404'   	  => array(
			'type'        => 'dropdown-pages',  
	 		'label'       => esc_attr__( 'Select Page', 'ronmi' ), 
	 		'description' => esc_attr__( 'Choose a template in pages.', 'ronmi' ), 
	 		'section'     => 'error_404', 
	 		'default'     => '', 
			 'priority'    => 3,
			 'active_callback' => array(
				array(
					'setting'  => 'custom_404',
					'operator' => '==',
					'value'    => 1,
				),
			),
		),

		/*color scheme*/
        'dark_light'  => array(
            'type'        => 'toggle',
            'label'       => esc_html__( 'Version Dark', 'ronmi' ),
            'section'     => 'color_scheme',
            'default'     => '',
            'priority'    => 3,
        ),
        'bg_body'      => array(
            'type'     => 'color',
            'label'    => esc_html__( 'Background Body', 'ronmi' ),
            'section'  => 'color_scheme',
            'default'  => '',
            'priority' => 10,
            'output'   => array(
                array(
                    'element'  => 'body, .site-content',
                    'property' => 'background-color',
                ),
            ),
        ),
        'main_color'   => array(
            'type'     => 'color',
            'label'    => esc_html__( 'Primary Color', 'ronmi' ),
            'section'  => 'color_scheme',
            'default'  => '#00776e',
            'priority' => 10,
        ),
        'second_color'   => array(
            'type'     => 'color',
            'label'    => esc_html__( 'Second Color', 'ronmi' ),
            'section'  => 'color_scheme',
            'default'  => '#9a644b',
            'priority' => 10,
        ),

        /*google atlantic*/
        'js_code'  => array(
            'type'        => 'code',
            'label'       => esc_html__( 'Code', 'ronmi' ),
            'section'     => 'script_code',
            'choices'     => [
				'language' => 'js',
			],
            'priority'    => 3,
        ),
		
	);
	$settings['panels']   = apply_filters( 'ronmi_customize_panels', $panels );
	$settings['sections'] = apply_filters( 'ronmi_customize_sections', $sections );
	$settings['fields']   = apply_filters( 'ronmi_customize_fields', $fields );

	return $settings;
}

$ronmi_customize = new Ronmi_Customize( ronmi_customize_settings() );
